% countOverTimeLED.m
%
%
%
%
% This is a script that I use to count pumping of the C. elegans
% pharynx.  It works like this:
% 
% > countOverTime
% > Name of strain: (N2)
% > Hit space bar to score worm 1's first condition...
% 
% After you hit space bar, the program will count down for as 
% many seconds specified, and then beep when it's finished.
% During this time, hit the space bar every time the worm pumps.
% you can score the light condition:
% 
% > Counted 25 pumps.  Hit enter to continue.
%
% This prompt is here so that spacebar
% presses after the elapsed time don't accidentally trigger the
% next condition (if there is one).
% 
% > Hit space bar to score worm 1's second condition...
% 
% This cycle will continue for all conditions, and then for the
% number of worms specified in the parameters below.  After this
% is all done, it will exclaim
%
% > Experiment complete! :)
%
% Then you can run it again to do another strain.
%
% The program will create a struct called 'strain' that will
% contain n fields, one for each condition.  It will save this
% struct to a file in your current directory (e.g. N2.mat).
%
% After all the data has been stored, run graphResults to 
% generate all sorts of fun graphs.  The graphs will compare
% conditions within strain (mean w/ SD) and across strains
% (mean w/o SD).

% PARAMETERS
% Edit the following before running the program to change various options.
lg = {'365 nm light - benchtop LED, physiology room'};     % Labels for graph legend
nWorms = 5         % Number of worms to score for the experiment.
startDelay = 10;    % Seconds of first condition (e.g. no light)
period = 10;        % Seconds of each subsequent condition
nTimes = 3;         % Number of subsequent conditions - each is marked with a beep
eventName = 'Pumps';% Use string of choice for "X per sec" in graph 
maxEvents = 20000;    % Max number of events to record for each worm

winWidth = 1000;    % Number of ms to use as a moving average in the graph
autoSave = true;    % Whether to automatically save the data to a file
genGraph = true;    % Graph will plot one line (mean +- SD) per condition
acquireData = true; % Turn to false if you just want to plot data
yMax = 6.5;

errset = 'se';      % switch between se and sd to change what error shading is plotted
errScale = 1;       % plot 2 standard errors

linecm = [255 51 0; 98 0 255] ./ 255;  % Color of line in graph and line of stim
patchcm = [255 214 126; 250 200 200] ./ 255;    % Color of shading


% CONSTANTS
ds = datestr(now, 'yyyymmdd');
numMsInS = 1000;
totalTime = startDelay + (nTimes - 1)*period;

% Start
begin = input('Have you checked all the parameters in countOverTime.m? (y/n) ', 's');
if (begin == 'n')
    disp('Restart after you have confirmed all params!');
    return;
end

strain = input('Name of strain: ', 's');

s = daq.createSession('ni');
ch = s.addAnalogOutputChannel('Dev1', 0, 'Voltage');

if (acquireData)
    data = cell(nWorms, 1);
    % Make my own for loop, so I can change the index within the loop
    w = 1;
    while(w <= nWorms)
        disp(['-Worm #' num2str(w) ':']); 
        vec = zeros(1,maxEvents);
        k = 'n';
        while(true)
            k = input('  Press ''c'' then RETURN to start the timer, ''r'' to redo last worm:', 's');
            if (k == 'c')
                break;
            elseif (k == 'r')
                if (w ~= 1) % Started new worm - go back
                    w = w - 1;
                end
                disp(['Redoing worm #' num2str(w)]);
                break;
            end
        end

        t = timer('TimerFcn', { @beepAndLightLED, s }, 'StopFcn', 'timeup=true;', ...
            'ExecutionMode', 'fixedSpacing', 'TasksToExecute', nTimes, ...
            'StartDelay', startDelay, 'Period', period);  % All time units in seconds
        start(t);
        tic

        timeup = false;
        nPresses = 0;
        while(timeup == false)
            pause;
            nPresses = nPresses + 1;
            vec(nPresses) = round(toc * numMsInS);  % Convert to ms for graphing
        end
        vec = vec(vec > 0);
        vec = vec(vec < totalTime * numMsInS);  % Convert to ms
        data{w} = vec;

        disp(['Events counted: ', num2str(length(vec))]);

        % Give a chance to redo the very last trial
        if (w == nWorms)
            while(true)
                k = input('Redo last trial of experiment? (y/n):', 's');
                if (k == 'y')
                    w = w - 1;
                    break;
                elseif (k == 'n')
                    break;
                end
            end
        end

        w = w + 1;
    end
end

varname = [strain '_' ds];
eval([varname ' = data;']);
if (autoSave && exist([varname '.mat'], 'file') == 0)
    save([varname '.mat'], varname, 'startDelay', 'period', 'nTimes','nWorms');
else
    disp('NOTE: Data not automatically saved because autosave=false or file already exists.');
end

% Get y-vals for the plots
[m sd se] = eventsOverTime(data, totalTime*numMsInS, winWidth, numMsInS, eventName, false);

if (strcmp(errset, 'se'))
    err = se * errScale;
else
    err = sd * errScale;
end

if(genGraph)
    f = figure;
    hold on
    x = winWidth/numMsInS/2:(1/numMsInS):totalTime-winWidth/numMsInS+winWidth/numMsInS/2;
    p = patch([x x(end:-1:1)], [m+err m(end:-1:1)-err(end:-1:1)], ...
        patchcm(1,:), 'LineStyle', 'none');
    plot(x, m, 'LineWidth', 2, 'Color', linecm(1,:));

    t = title([eventName ' r sec (' varname '_dissecting scope n=' int2str(nWorms) ') '], 'FontWeight', 'bold', 'FontSize', 16);
    set(t, 'interpreter', 'none');  % Turn off _ causing subscripts
    xlabel('time (s)', 'FontWeight', 'bold', 'FontSize', 12);
    ylabel([eventName ' per sec'], 'FontWeight', 'bold', 'FontSize', 12);
    yts = 0:1:yMax;
    set(gca, 'YTick', yts); 
    prettify();
    axis([winWidth/numMsInS/2 totalTime-winWidth/numMsInS+winWidth/numMsInS/2 0 yMax]);
    
    % Plot time during which light is on
    h = plot([startDelay startDelay+period], [yMax-.5 yMax-.5], 'LineWidth', 4, 'Color', linecm(2,:));
    
    %legend(h, lg);
    text(12.5, yMax-.35, lg, 'FontWeight', 'bold');
    
    saveas(f, [varname '.fig']);
end

s.delete(); %this command ends the session, freeing the hardware to be used again. Very important!